<?php
namespace App\Services;

use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DynamicArchiveService
{
    public function archiveOldRecords($olderThanMonths = 12)
    {
        // Get all tables in the database
        $tables = DB::select("SHOW TABLES");

        foreach ($tables as $table) {
            $tableName = reset($table); // Get table name dynamically

            // Skip the archive table itself
            if ($tableName === 'generic_archives') {
                continue;
            }

            // Check if the table has a 'created_at' column
            $columns = DB::getSchemaBuilder()->getColumnListing($tableName);
            if (!in_array('created_at', $columns)) {
                \Log::info("No 'created_at' column in table: $tableName");
                continue;
            }

            // Fetch old records
            $oldRecords = DB::table($tableName)
                ->where('created_at', '<', Carbon::now()->subMonths($olderThanMonths))
                ->get();

            if ($oldRecords->isEmpty()) {
                \Log::info("No old records found for table: $tableName");
            } else {
                \Log::info("Found " . $oldRecords->count() . " old records for table: $tableName");
            }

            foreach ($oldRecords as $record) {
                // Log the record before archiving
                \Log::info("Archiving record from table: $tableName with ID: " . $record->id);

                // Enable query log
                DB::enableQueryLog();

                // Archive the record
                DB::table('generic_archives')->insert([
                    'table_name' => $tableName,
                    'data' => json_encode($record),
                    'created_at' => now(),
                ]);

                // Log the query executed for archiving
                \Log::info('Query Log for Insert:', DB::getQueryLog());

                // Delete from the original table
                DB::table($tableName)->where('id', $record->id)->delete();

                // Log the deletion of the record
                \Log::info("Deleted record from table: $tableName with ID: " . $record->id);
            }
        }
    }
}