<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PlanController extends Controller
{
    public function index(Request $request)
    {
        Log::info('Fetching all plans', ['search' => $request->search]);

        $query = Plan::orderBy('created_at', 'desc');
        if ($request->has('search') && !empty($request->search)) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }
        $plans = $query->paginate(20);

        return view('admin.plans.all-plans', compact('plans'));
    }

    public function create()
    {
        Log::info('Opening create plan form');
        return view('admin.plans.add-plan');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'price' => 'required|numeric',
            'duration_days' => 'required|integer',
            'max_ads' => 'required|integer',
            'max_media' => 'required|integer',
            'is_premium' => 'required|in:0,1',
        ]);

        $plan = Plan::create($request->all());
        Log::info('New plan created', ['plan_id' => $plan->id]);

        return redirect()->route('admin.plans')->with('success', 'Plan created successfully.');
    }

    public function edit($id)
    {
        $plan = Plan::findOrFail($id);
        Log::info('Editing plan', ['plan_id' => $id]);

        return view('admin.plans.plans-edit', compact('plan'));
    }

    public function update(Request $request, $id)
    {
        $plans = Plan::findOrFail($id);

        $request->validate([
            'name' => 'required',
            'stripe_price_id' => 'required',
            'price' => 'required|numeric',
            'duration_days' => 'required|integer',
            'max_ads' => 'required|integer',
            'max_media' => 'required|integer',
            'is_premium' => 'required|in:0,1',
        ]);

        $plans->update([
            'name' => $request->name,
            'stripe_price_id' => $request->stripe_price_id,
            'price' => $request->price,
            'duration_days' => $request->duration_days,
            'max_ads' => $request->max_ads,
            'max_media' => $request->max_media,
            'is_premium' => $request->is_premium,
        ]);

        Log::info('Plan updated', ['plan_id' => $id]);

        return redirect()->route('admin.plans')->with('success', 'Plan Updated successfully.');
    }

    public function destroy($id)
    {
        $plan = Plan::find($id);
        if (!$plan) {
            Log::warning('Plan not found for deletion', ['plan_id' => $id]);
            return redirect()->route('admin.plans')->with('error', 'Plan not found.');
        }

        $plan->delete();
        Log::info('Plan deleted', ['plan_id' => $id]);

        return redirect()->route('admin.plans')->with('success', 'Plan deleted successfully.');
    }
}
