<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SellerPayment;
use App\Models\Setting;
use App\Models\Ad;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Checkout\Session;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function addpromote($id)
    {
        $settings = Setting::all();
        $ad = Ad::findOrFail($id);
        return view("seller.pay", compact("settings", "ad"));
    }

    public function processPayment(Request $request, $id)
    {
        Log::info('Processing payment for ad id: ' . $id);
        $request->validate([
            'fee' => 'required|numeric|min:1',
            'duration' => 'required|integer|min:1',
        ]);
        $fee = ($request->input('fee'));
        $duration = ($request->input('duration'));

        Log::info('Received fee: ' . $fee . ' and duration: ' . $duration);

        $ad = Ad::findOrFail($id);
        $selectedSetting = Setting::where('promotion_fee', $fee)->first();

        if (!$selectedSetting) {
            Log::error('Invalid fee selected for ad id: ' . $id);
            return redirect()->back()->with('error', 'Invalid fee selected!');
        }
        $promotionFee = $selectedSetting->promotion_fee;
        Stripe::setApiKey(env('STRIPE_SECRET'));

        $successUrl = url(route('payment.success')) . '?sesion_id={CHECKOUT_SESSION_ID}&ad_id=' . $ad->id . '&promotion_fee=' . $promotionFee . '&duration=' . $duration;
        Log::info('Generated Success URL: ' . $successUrl);

        $session = Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
                'price_data' => [
                'currency' => 'usd',
                'product_data' => ['name' => 'Ad Promotion', 'description' => 'Duration: ' . $duration],
                'unit_amount' => $promotionFee * 100,
            ],
            'quantity' => 1,
        ]],
        'mode' => 'payment',
        'success_url' => $successUrl,
        'cancel_url' => route('payment.cancel'),
    ]);

    Log::info('Payment session created for ad id: ' . $id . ' | Session ID: ' . $session->id);

    return redirect($session->url);
}



    public function paymentSuccess(Request $request)
    {
        Log::info('Payment success for ad id: ' . $request->ad_id);
        Log::info('Received session_id: ' . $request->sesion_id);

        if (!auth()->check()) {
            return redirect('/login')->with('error', 'You need to be logged in to make a payment.');
        }

        $ad = Ad::findOrFail($request->ad_id);
        $promotionFee = round($request->input('promotion_fee'), 2);
        $duration = $request->input('duration');

        Log::info('Received promotion fee: ' . $promotionFee);
        Log::info('Received duration: ' . $duration);

        if ($promotionFee == 0 || !in_array($promotionFee, [90.00, 80.00, 70.00, 400.00])) {
            Log::error('Invalid promotion fee selected.');
            return redirect('/dashboard')->with('error', 'Invalid promotion fee selected.');
        }

        Log::info('Available promotion fees in settings: ' . json_encode(Setting::pluck('promotion_fee')->toArray()));

        $selectedSetting = Setting::where('promotion_fee', $promotionFee)->first();

        if (!$selectedSetting) {
            Log::error('Invalid promotion fee selected.');
            return redirect('/dashboard')->with('error', 'Invalid promotion fee selected.');
        }

        $promotionDuration = (int) $duration ?? (int) $selectedSetting->promotion_duration ?? 30;

        Stripe::setApiKey(env('STRIPE_SECRET'));

        $sessionId = $request->input('sesion_id');

        if (!$sessionId) {
            Log::error('Missing session_id on payment success page.');
            return redirect('/ads')->with('error', 'Payment session ID is missing.');
        }

        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));
        $session = $stripe->checkout->sessions->retrieve($sessionId);

        Log::info('Stripe Session Object: ' . json_encode($session));
        Log::info('Stripe Session ID: ' . $session->id);

        $paymentMethodType = 'Unknown';
        if (!empty($session->payment_intent)) {
            try {
                $paymentIntent = \Stripe\PaymentIntent::retrieve($session->payment_intent);
                $paymentMethod = \Stripe\PaymentMethod::retrieve($paymentIntent->payment_method ?? '');
                $paymentMethodType = $paymentMethod->type ?? 'Unknown';
            }   catch (\Exception $e) {
                Log::error('Payment method retrieval failed: ' . $e->getMessage());
            }
        }

        $ad->is_promoted = true;
        $ad->promotion_expiry = now()->addDays($promotionDuration);
        $ad->save();

        SellerPayment::create([
            'user_id' => auth()->id(),
            'ad_id' => $ad->id,
            'amount' => floatval($selectedSetting->promotion_fee),
            'payment_status' => 'success',
            'transaction_id' => $session->id,
            'payment_method' => $paymentMethodType,
            'duration' => $promotionDuration,
        ]);

        Log::info('Payment recorded for ad id: ' . $ad->id . ' | Transaction ID: ' . $session->id);
        return redirect('/dashboard')->with('success', 'Your ad has been promoted successfully!');
    }

    public function paymentCancel()
    {
        Log::info('Payment canceled for ad id: ' . request('ad_id'));
        return redirect('/dashboard');
    }
}
