<?php

namespace App\Http\Controllers;

use App\Models\Ad;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\SellerActivityLog;
use App\Models\verification;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Hash;
use App\Rules\ReCaptcha;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use App\Jobs\DeleteUnverifiedUser;
use App\Services\TwilioService;

class UserController extends Controller
{

    protected $twilio;

    public function __construct(TwilioService $twilio){
        $this->twilio = $twilio;
    }

    public function register(Request $request)
{

    Log::info('Register method accessed');

    // Validation
    $request->validate([
        'email' => 'required|string|email|max:255|unique:users',
        'name' => 'required|string|max:255',
        'password' => 'required|string|confirmed|min:8',
        'password_confirmation' => 'required',
        'phone_number' => 'required|regex:/^\+?[0-9]{10,15}$/|unique:users',
        'is_adult' => 'required',
        'is_disease_free' => 'required',
        'is_drug_free' => 'required',
        'is_agree' => 'required',
        // 'g-recaptcha-response' => ['required', new ReCaptcha]
    ]);

    $user = User::create([
        'name' => $request->name,
        'email' => $request->email,
        'phone_number' => $request->phone_number,
        'is_adult' => $request->is_adult,
        'is_disease_free' => $request->is_disease_free,
        'is_drug_free' => $request->is_drug_free,
        'is_agree' => $request->is_agree,
        'password' => bcrypt($request->password),
    ]);

    $otp = rand(1000, 9999);
    Log::info('Generated OTP', ['otp' => $otp]);
    $user->otp = $otp;
    $user->otp_expires_at = now()->addMinutes(5);
    $user->save();

    try {
        $this->twilio->sendSms($user->phone_number, "Your OTP Code is: $otp (Valid for 5 minutes)");
    } catch (\Exception $e) {
        Log::error("Twilio SMS Error: " . $e->getMessage());
    }

    Auth::login($user);

    return redirect()->route('otp.verify.form');
}

    public function showOtpForm()
    {
        return view('seller.otp');
    }

    public function verifyOtp(Request $request)
    {
        Log::info('OTP verification requested:', ['user_id' => Auth::id(), 'otp' => $request->otp]);
    
        $request->validate([
            'otp' => 'required|array|size:4',
            'otp.*' => 'required|digits:1'
        ]);
    
        $user = Auth::user();
    
        if (!$user) {
            return back()->with(['error' => 'User not found.']);
        }
    
        $otpFromUser = implode('', $request->otp);
        $storedOtp = (string) $user->otp;
        $otpCreatedAt = Carbon::parse($user->otp_expires_at);
    
        if (now()->greaterThan($otpCreatedAt)) { 
            Log::warning('OTP expired for user ID: ' . $user->id);
            $user->delete();
            return redirect()->route('register')->with(['error' => 'OTP expired, please register again.']);
        }
    
        if ($otpFromUser === $storedOtp) {
            Log::info('OTP matched for user ID: ' . $user->id);
            $user->otp = null;
            $user->otp_expires_at = null;
            $user->save();
    
            return redirect()->route('login')->with(['success' => 'Phone Number verified successfully.']);
        } else {
            Log::warning('Invalid OTP attempt for user ID: ' . $user->id);
            return back()->with(['error' => 'Invalid OTP, please try again.']);
        }
    }
    
    
    public function login(Request $request)
{
    $credentials = $request->only('email', 'password');

    if (Auth::attempt($credentials)) {
        $seller = Auth::user();

        // Check if the seller is blocked
        if ($seller->status == "blocked") {  // Assuming 'blocked' is a boolean column in your 'users' or 'sellers' table
            Auth::logout();
            return back()->withErrors([
                'email' => 'Your account is blocked. Please contact support.',
            ])->withInput();
        }

        $seller->update(['last_active_at' => now()]);

        if ($seller->role === 'seller') {
            SellerActivityLog::create([
                'seller_id' => $seller->id,
                'ip_address' => $request->ip(),
                'location' => $this->getLocation($request->ip()),
                'browser_details' => $request->header('User-Agent'),
                'activity_type' => 'Login',
            ]);

            $this->logActivity($seller->id, 'Logged In', 'User logged in from ' . $request->ip());
        }

        $request->session()->regenerate();

        if ($seller->role === 'admin') {
            return redirect()->route('admin.dashboard');
        }

        return redirect('/profile')->with('success', 'Welcome, Seller!');
    }

    return back()->withErrors([
        'email' => 'The provided credentials do not match our records.',
    ])->withInput();
}


    public function logout(Request $request){
        Auth::logout();
        $request->session()->invalidate();
        return redirect('/');
    }
    
    
    private function getLocation($ip)
    {
        if ($ip == '127.0.0.1') {
            return 'Localhost - Unknown Location';
        }
    
        $response = @file_get_contents("http://ip-api.com/json/{$ip}");
        $data = json_decode($response);
        
        if ($data && $data->status == "success") {
            return "{$data->city}, {$data->country}";
        }
    
        return "Unknown Location";
    }
    

    public function get_seller_address()
    {
        $getaddress = SellerActivityLog::with('seller')
        ->orderBy('activity_time','desc')
        ->paginate(5);
        return view('admin.seller_activity_log', compact('getaddress'));
    }

     public function update_password( Request $request){
        $profile = Auth::user()->profile;
        $validated = $request->validate([
            'old_password' => 'required|current_password',
            'password' => 'required|confirmed',
        ]);
        $user = Auth::user();
        $user->password = Hash::make($request->password);
        $result = $user->save();
        if( $result ){
            return view('profile_components.change_password', ['profile' => $profile, 'message' => 'Password updated successfully..!']);
        }else{
            return view('profile_components.change_password', ['profile' => $profile, 'message' => 'Something went wrong, Please try again.!']); 
        }
        
    }



    public function verification(Request $request){
        
        $profile = Auth::user()->profile;
        $validated = $request->validate([
            'images' => 'required|file|mimes:jpg,jpeg,png,gif',
        ]);

        $filePaths = [];
        $user = verification::where('user_id',Auth::user()->id)->get();
        if($user->count() > 0){
           foreach ($user as $item) {
                unlink(public_path('storage/'.$item->file_path));
           }
           verification::where('user_id',Auth::user()->id)->delete(); 
        }
        
        $isVerified = $profile->status == 'verified' ? true : false;
        $image = 'images/'.time().'.'.$request->images->getClientOriginalExtension();
        $request->images->move(public_path('storage/images'), $image);
        $filePaths[] = $image;
        $ad = new Verification(['file_path' => $image]);
        $ad->user_id = auth()->id();
        $ad->save();

        return view('profile_components.verification', ['images' => $filePaths, 'profile' => $profile, 'isVerified' => 0,'message' => 'Files uploaded successfully..!']);
    }
            

     function logActivity($userId, $action, $details = null)
    {
        ActivityLog::create([
            'user_id' => $userId,
            'action' => $action,
            'details' => $details
        ]);
    }
}