<?php

namespace App\Http\Controllers;

use App\Models\Accounts;
use App\Models\Ad;
use App\Models\Message;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\Profile;
use App\Models\Sellerad;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Stripe\PaymentIntent;
use Stripe\Account;
use Stripe\AccountLink;
use Stripe\Stripe;
use Stripe\Balance;
use Stripe\Payout;
class ProfileController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $profiles = Profile::all();
        return view("admin.profiles", ["profiles"=> $profiles]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if(Auth::user()->profile){
            return redirect('/profile')->with('error','Profile already created');
        }
        return view("seller.profile.create");   
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // dd($request->all());
        $validated = $request->validate([
            'gender'=> 'required',
            'age' => 'required|integer',
            'type'=> 'required|string',
            'clothing_size'=> 'nullable|string',
            'figure' => 'required|string',
            'hair_color' => 'required|string',
            'eye_color' => 'required|string',
            'nickname' => 'required|string',
            'address' => 'required|string',
            'instagram' => 'nullable|string|max:255', 
            'snapchat' => 'nullable|string|max:255', 
            'onlyfans' => 'nullable|string|max:255', 
            'description' => 'required|string',
            'profile_picture' => 'required|image'
        ]);

        // $profilePicPath = $request->file('profile_picture')->store('avatars', 'public');
        
        $profilePicPath = time().'.'.$request->profile_picture->getClientOriginalExtension();
        $request->profile_picture->move(public_path('storage/avatars'), $profilePicPath);
        $validated['profile_picture'] = $profilePicPath;
        $profile = new Profile($validated);
        $profile->user_id = auth()->id();
        $profile->save();
        if(auth()->user()->role === 'admin'){
            return redirect()->route('admin.dashboard');
        }
        return redirect('/profile')->with('success','Profile created successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show()
    {
        $profile = Auth::user()->profile;
        if(!$profile){
            return redirect()->route('profile.create');
        }
        // dd($profile);
        return view('profile_components.user_information', ['profile' => $profile]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Profile $profile)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
        public function update(Request $request, Profile $profile)
    {
        $validated = $request->validate([
            'age' => 'required|integer',
            'type' => 'required|string|max:50',
            'clothing_size' => 'required|string|max:255', 
            'figure' => 'required|string|max:50',
            'hair_color' => 'required|string|max:50',
            'eye_color' => 'required|string|max:50',
            'nickname' => 'required|string|max:255',
            'instagram' => 'nullable|string|max:255', 
            'snapchat' => 'nullable|string|max:255', 
            'onlyfans' => 'nullable|string|max:255', 
            'description' => 'nullable|string',
            'address' => 'required|string|max:255',
            'profile_picture' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'show_age' => 'nullable|boolean',
            'show_clothing_size' => 'nullable|boolean',
        ]);
    
        $validated['show_age'] = $request->has('show_age') ? true : false;
        $validated['show_clothing_size'] = $request->has('show_clothing_size') ? true : false;

        unset($validated['gender']);
    
        if ($request->hasFile('profile_picture')) {
            if ($profile->profile_picture && Storage::disk('public')->exists($profile->profile_picture)) {
                Storage::disk('public')->delete($profile->profile_picture);
            }
            $profilePicPath = time().'.'.$request->profile_picture->getClientOriginalExtension();
            $request->profile_picture->move(public_path('storage/avatars'), $profilePicPath);
            $validated['profile_picture'] = $profilePicPath;
        }
    
        $profile->update($validated);
    
        return redirect()->back()->with('success', 'Profile updated successfully');
    }
    
    public function imageUpdate(Request $request, Profile $profile){
        $validated = $request->validate([
            'profile_picture' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048'
        ]);

        if ($request->hasFile('profile_picture')) {
            if ($profile->profile_picture && Storage::disk('public')->exists($profile->profile_picture)) {
                Storage::disk('public')->delete($profile->profile_picture);
            }
            $profilePicPath = time().'.'.$request->profile_picture->getClientOriginalExtension();
            $request->profile_picture->move(public_path('storage/avatars'), $profilePicPath);
            $validated['profile_picture'] = $profilePicPath;
        }

        $profile->update($validated);

        return redirect()->back()->with('success', 'Profile updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Profile $profile)
    {
        //
    }

    public function verification()
    {
        $profile = Auth::user()->profile;

        if (!$profile) {
            return back()->with('error', 'Profile not found.');
        }
        $isVerified = $profile->status == 'verified' ? true : false;

        $images[] = $profile?->verification?->file_path;
        return view('profile_components.verification', [
            'profile' => $profile,
            'isVerified' => $isVerified,
            'images' => $images
        ]);
    }
    

    public function change_password(){
        $profile = Auth::user()->profile;
        return view('profile_components.change_password', ['profile' => $profile]);
    }
    public function read_messages(){
        $profile = Auth::user()->profile;
        $messages = Message::where('user_id', Auth::id())->orderBy('id', 'desc')->paginate(10);
        return view('profile_components.read_messages', ['profile' => $profile,'messages' => $messages]);
    }


    public function user_information(){
        $profile = Auth::user()->profile;
        return view('profile_components.user_information', ['profile' => $profile]);
    }

    public function balance(){
    $profile = Auth::user()->profile;
    Stripe::setApiKey(env('STRIPE_SECRET'));

    // Check if user has already created a Stripe account
    $seller_account = Accounts::where('user_id', Auth::id())->first();
    if (!$seller_account) {
        try {
            // Create a new Express account
            $account = Account::create([
                'type' => 'express',
                'country' => 'US',
                'email' => Auth::user()->email,
            ]);

            // Store the new Stripe account ID
            Accounts::create([
                'user_id' => Auth::id(),
                'account_number' => $account->id,
            ]);

            // Generate an account onboarding link
            $accountLink = AccountLink::create([
                'account' => $account->id,
                'refresh_url' => route('profile.balance'),
                'return_url' => route('profile.dashboard'),
                'type' => 'account_onboarding',
            ]);
           
            return view('profile_components.balance', [
                'profile' => $profile,
                'accountLink' => $accountLink->url
            ]);
        } catch (\Exception $e) {
            return view('profile_components.balance', ['profile' => $profile, 'error' => $e->getMessage()]);
        }
    }

    try {
        // Retrieve Stripe balance
        if (!$seller_account->account_number) {
            return view('profile_components.balance', ['profile' => $profile, 'error' => 'Stripe account ID missing']);
        }

        $balance = Balance::retrieve([
            'stripe_account' => $seller_account->account_number,
        ]);

        $currentBalance = collect($balance->available)->sum('amount');
        $frozenBalance = collect($balance->pending)->sum('amount');

        // Retrieve last payout
        $payouts = Payout::all(['limit' => 1], ['stripe_account' => $seller_account->account_number]);
        $lastPayoutAmount = $payouts->data[0]->amount ?? 0;
        $lastPayoutDate = $payouts->data[0]->created ?? null;

        // Calculate total earnings
        $user = Auth::user();
        $totalEarnings = $user->ads->flatMap(function ($ad) {
            return $ad->orders->flatMap(function ($order) {
                return $order->orderDetail->customer_payment;
            });
        })->sum('amount');

        // Format response
        $details = [
            'current_balance' => $currentBalance / 100,
            'last_payout_amount' => $lastPayoutAmount / 100,
            'last_payout_date' => $lastPayoutDate ? date('Y-m-d H:i:s', $lastPayoutDate) : null,
            'frozen_balance' => $frozenBalance / 100,
            'total_earnings' => $totalEarnings,
        ];

        return view('profile_components.balance', ['profile' => $profile, 'details' => $details]);
    } catch (\Stripe\Exception\ApiErrorException $e) {
        return view('profile_components.balance', ['profile' => $profile, 'error' => $e->getMessage()]);
    }
}


public function add_seller(){
    $profile = Auth::user()->profile;
    Stripe::setApiKey(env('STRIPE_SECRET'));

    // Check if user has already created a Stripe account
    $seller_account = Accounts::where('user_id', Auth::id())->first();
    if (!$seller_account) {
        try {
            // Create a new Express account
            $account = Account::create([
                'type' => 'express',
                'country' => 'US',
                'email' => Auth::user()->email,
            ]);

            // Store the new Stripe account ID
            Accounts::create([
                'user_id' => Auth::id(),
                'account_number' => $account->id,
            ]);

            // Generate an account onboarding link
            $accountLink = AccountLink::create([
                'account' => $account->id,
                'refresh_url' => route('profile.balance'),
                'return_url' => route('profile.dashboard'),
                'type' => 'account_onboarding',
            ]);
           
            return view('profile_components.balance', [
                'profile' => $profile,
                'accountLink' => $accountLink->url
            ]);
        } catch (\Exception $e) {
            return view('profile_components.add-seller-account', ['profile' => $profile, 'error' => $e->getMessage()]);
        }
    }
}

    public function onPageRloadCreateAccountNumber(Request $request)
{
    Stripe::setApiKey(env('STRIPE_SECRET'));

    try {
        // Create a Stripe Express account for the seller
        $stripeAccount = Account::create([
            'type' => 'express',
            'country' => 'US', // Change as per your country
            'email' => Auth::user()->email,
            'capabilities' => [
                'card_payments' => ['requested' => true],
                'transfers' => ['requested' => true],
            ],
        ]);


        // Store the seller account in your database
        $newAccount = Accounts::create([
            'user_id' => Auth::id(),
            'account_name' => $request->account_holder_name,
            // 'account_number' => $stripeAccount->id, // Store Stripe Account ID
            'method' => $request->account_type,
        ]);

        return redirect()->back()->with('success', 'Stripe account created successfully!');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', $e->getMessage());
    }
}

    
    public function orders(){
        $profile = Auth::user()->profile;
        $orders = DB::table('ads')
        ->join('orders', 'orders.ad_id', '=', 'ads.id')
        ->join('order_details', 'orders.order_detail_id', '=', 'order_details.id')
        ->where('ads.user_id', Auth::id())
        ->select('orders.*','orders.id as order_id','orders.status as order_status', 'ads.*', 'order_details.*')
        ->orderBy('orders.id', 'desc')
        ->paginate(10);
        return view('profile_components.orders', ['profile' => $profile,'orders' => $orders]);
    }


    public function offers(){
        $profile = Auth::user()->profile;
        $offers  = Sellerad::where('user_id', Auth::id())
        ->where('offer_status','Accepted')
        ->paginate(10);
        // return $offers;
        return view('profile_components.seller-offers', ['profile' => $profile,'offers' => $offers]);
    }
   public function add_link(Request $request){
        $request->validate([
           'tracking_link' => 'required'
        ]);
        Sellerad::where('id', $request->offer_id)
        ->update([
            'tracking_link' => $request->tracking_link
        ]);
        return redirect()->back();
   }

   public function offer_sent_status(Request $request){   
        Sellerad::where('id', $request->offer_id)
        ->update([
            'offer_sent_status' => $request->offer_sent_status
        ]);
        return redirect()->back();
   }

   public function message(){
    return view('seller.message');
   }

   public function sendmessage(Request $request)
{
    $request->validate([
        'subject' => 'required',
        'message' => 'required'
    ]);

    $user_id = auth()->id();

    $message = Message::create([
        'subject' =>$request->subject,
        'message' => $request->message,
        'user_id' => $user_id
    ]);

    if ($message) {
        return redirect('/profile')->with('success','Message sent successfully');
    } else {
        return redirect('profile.message')->with('error','Message not sent');
    }
}

}
