<?php

use Illuminate\Support\Facades\Route;
use Stripe\Stripe;
use Stripe\Account;
use Stripe\PaymentIntent;
use Stripe\Transfer;
use Stripe\Refund;

// Step 1: Configure Stripe API Key
Stripe::setApiKey(env('STRIPE_SECRET'));

// Step 2: Onboard Seller (Create a Connected Account)
Route::post('/onboard-seller', function (Request $request) {
    $sellerEmail = $request->input('email');

    $account = Account::create([
        'type' => 'express',
        'country' => 'US',
        'email' => $sellerEmail,
    ]);

    return response()->json([
        'status' => 'success',
        'account_id' => $account->id,
        'link' => $account->url // URL for completing the onboarding
    ]);
});

// Step 3: Buyer Makes a Payment
Route::post('/make-payment', function (Request $request) {
    $amount = $request->input('amount'); // e.g., 100 USD -> amount = 10000 (cents)
    $platformFee = $request->input('platform_fee'); // e.g., 2% of the amount
    $sellerAccountId = $request->input('seller_account_id'); // Connected Seller Stripe ID

    $paymentIntent = PaymentIntent::create([
        'amount' => $amount,
        'currency' => 'usd',
        'payment_method_types' => ['card'],
        'application_fee_amount' => $platformFee, // Platform Fee
        'transfer_data' => [
            'destination' => $sellerAccountId, // Seller Connected Account
        ],
    ]);

    return response()->json([
        'status' => 'success',
        'client_secret' => $paymentIntent->client_secret, // ed for confirming the payment on the frontendUs
    ]);
});

// Step 4: Transfer Funds After Buyer and Seller Confirm Delivery
Route::post('/confirm-transaction', function (Request $request) {
    $amount = $request->input('amount'); // Amount to transfer to seller after deductions
    $sellerAccountId = $request->input('seller_account_id');

    $transfer = Transfer::create([
        'amount' => $amount, // Amount in cents
        'currency' => 'usd',
        'destination' => $sellerAccountId, // Seller's Stripe Account
        'description' => 'Payout for transaction #12345',
    ]);

    return response()->json([
        'status' => 'success',
        'transfer_id' => $transfer->id,
    ]);
});

// Step 5: Handle Refunds (Optional)
Route::post('/refund', function (Request $request) {
    $paymentIntentId = $request->input('payment_intent_id');

    $refund = Refund::create([
        'payment_intent' => $paymentIntentId,
    ]);

    return response()->json([
        'status' => 'success',
        'refund_id' => $refund->id,
    ]);
});

// Step 6: Seller Dashboard with Withdrawal Requests
Route::get('/seller-dashboard', function () {
    // Fetch transaction details for a seller from your database
    $transactions = DB::table('transactions')
        ->where('seller_id', Auth::id())
        ->get();

    return view('seller.dashboard', [
        'transactions' => $transactions,
    ]);
});

Route::post('/request-withdrawal', function (Request $request) {
    $amount = $request->input('amount');
    $sellerAccountId = $request->input('seller_account_id');

    $transfer = Transfer::create([
        'amount' => $amount * 100, // Convert to cents
        'currency' => 'usd',
        'destination' => $sellerAccountId,
        'description' => 'Seller withdrawal request',
    ]);

    // Update transaction status in your database
    DB::table('transactions')->insert([
        'seller_id' => Auth::id(),
        'amount' => $amount,
        'status' => 'completed',
        'created_at' => now(),
    ]);

    return response()->json([
        'status' => 'success',
        'transfer_id' => $transfer->id,
    ]);
});
