<?php

namespace App\Http\Controllers;

use App\Mail\OrderUpdateMail;
use App\Models\Order;
use App\Models\DocumentMedia;
use App\Models\OrderDetail;
use Illuminate\Auth\Events\Validated;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Mail\OrderInProgressEmail;
use App\Mail\OrderCanceledEmail;
use App\Mail\OrderSentEmail;
use Illuminate\Support\Facades\Mail;


class OrderController extends Controller
{

    public function update_order_status(Request $request)
{
    Log::info('Update Order Status Request:', $request->all());
    // Find the order by order_detail_id
    $order = Order::where('order_detail_id', $request->id)->first();
    
    // Check if order exists
    if (!$order) {
        Log::error('Order not found: ' . $request->id);
        return redirect()->back()->with('error', 'Order not found.');
    }

    // Get the associated order detail (to access the email)
    $orderDetail = $order->orderDetail; // This will get the order detail associated with the order

    // Check if order detail exists and has an email
    if (!$orderDetail || !$orderDetail->email) {
        Log::error('Order detail or email not found for Order ID: ' . $request->id);
        return redirect()->back()->with('error', 'Order detail or email not found.');
    }

    // Log order and order detail for debugging
    Log::info('Order:', [$order]);
    Log::info('Order Detail:', [$orderDetail]);

    // Store old status
    $oldStatus = $order->status;

    // Update the order status
    $order->update([
        'status' => $request->status,
    ]);

    Log::info('Order status updated successfully.');

    // Check if the status is changed to 'inprogress' and send email
    if ($request->status == 'inprogress' && $oldStatus != 'inprogress') {
        Log::info('Sending in progress email...');
        $this->sendOrderInProgressEmail($orderDetail->email, $order);
    }

    Log::info('Request Status: ' . $request->status);
    Log::info('Old Status: ' . $oldStatus);

    // Check if the status is changed to 'canceled' and send cancellation email
    if ($request->status == 'cancelled' && $oldStatus != 'cancelled') {
        Log::info('Sending canceled order email...');
        $this->sendOrderCanceledEmail($orderDetail->email, $order);
    }

        // Check if the status is changed to 'sent' and send sent email
        if ($request->status == 'sent' && $oldStatus != 'sent') {
            Log::info('Sending sent order email...');
            $this->sendOrderSentEmail($orderDetail->email, $order, $request->pdf, $request->media);

        }
      

    return redirect()->back()->with('success', 'Order status updated.');
}


protected function sendOrderInProgressEmail($email, $order)
{
    // Send email using Laravel's mail system for in progress
    Mail::to($email)->send(new OrderInProgressEmail($order));
}

protected function sendOrderCanceledEmail($email, $order)
{
    Log::info('Sending cancellation email to: ' . $email);

    // Send email using Laravel's mail system for canceled order
    Mail::to($email)->send(new OrderCanceledEmail($order));
}

protected function sendOrderSentEmail($email, $order, $pdf = null, $media = null)
{
    Log::info('Sending email to: ' . $email);

    Log::info('All Request Data:', request()->all());
    Log::info('Uploaded Files:', request()->file());

    $pdfPath = $pdf ? $pdf->store('order-docs', 'public') : null;
    Log::info('PDF Path: ' . ($pdfPath ?? 'No PDF uploaded'));
    
    $videoPath = $media ? $media->store('media/snippets', 'public') : null;
    

    // Store file paths in `document_media` table
    DocumentMedia::create([
        'pdf' => $pdfPath,
        'media' => $videoPath,
    ]);

    // Send email with attachments
    Mail::to($email)->send(new OrderSentEmail($order, $pdfPath, $videoPath));
}



}
