<?php

namespace App\Http\Controllers;

use App\Models\Accounts;
use App\Models\Ad;
use App\Models\Contact;
use App\Models\CustomerPayment;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\MediaFile;
use App\Models\Category;
use Stripe\Stripe;
use Stripe\Charge;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Stripe\PaymentIntent;
use App\OrderConfirmationTrait;
use Stripe\Transfer;

class CustomerController extends Controller
{
    use OrderConfirmationTrait;

    public function checkout(Request $request){
        $request->validate([
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'email' => 'required|email',
            // 'phone_number' => 'nullable|integer',
            'address' => 'required|string',
            'city'=> 'required|string',
            'postcode' => 'required|string',
            'quantity.*' => 'required|integer|min:1'
        ]);
        
        $total_price = 0;
        foreach ($request->quantity as $key => $value) {
            $total_price += $value * $request->price[$key];
        }
       
        return view('user.checkout',['data'=>$request->all(),'total_price'=>$total_price]);
    }

    public function payment(Request $request) {
        $data = json_decode($request->data);
    
        $order_datails = OrderDetail::create([
            'first_name' => $data->first_name,
            'last_name' => $data->last_name,
            'email' => $data->email,
            'phone_number' => $data->phone_number,
            'address' => $data->address,
            'city' => $data->city,
            'postcode' => $data->postcode,
        ]);
    
        $order_datails_id = $order_datails->id;
    
        try {
            Stripe::setApiKey(env('STRIPE_SECRET'));
    
            $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));
            $paymentMethod = $stripe->paymentMethods->create([
                'type' => 'card',
                'card' => [
                    'token' => $request->stripeToken,
                ],
            ]);
    
            $customer = $stripe->customers->create([
                'email' => $data->email,
                'name' => $data->first_name,
            ]);
    
            $stripe->paymentMethods->attach(
                $paymentMethod->id,
                ['customer' => $customer->id]
            );
    
            foreach ($data->ids as $key => $ad_id) {
                $ad = Ad::find($ad_id);
    
                if (!$ad) {
                    Log::error("Ad not found for ad_id: " . $ad_id);
                    return redirect('/cart')->with('error', 'Ad not found.');
                }
    
                Log::info('Processing payment for Ad User ID: ' . $ad->user_id);
    
                $sellerAccount = Accounts::where('user_id', $ad->user_id)->first();
    
                if (!$sellerAccount) {
                    $sellerAccount = Accounts::where('id', 1)->first();

                }
    
                $sellerAccountId = $sellerAccount->account_number;
    
                // $paymentIntent = PaymentIntent::create([
                //     'amount' => $data->quantity[$key] * $data->price[$key] * 100,
                //     'currency' => 'usd',
                //     'customer' => $customer->id,
                //     'payment_method' => $paymentMethod->id,
                //     'capture_method' => 'manual',
                //     'application_fee_amount' => 2,
                //     'on_behalf_of' => $sellerAccountId,
                //     'transfer_data' => ['destination' => $sellerAccountId],
                //     'automatic_payment_methods' => [
                //         'enabled' => true,
                //         'allow_redirects' => 'never',
                //     ],
                // ]);


                $paymentIntent = PaymentIntent::create([
                    'amount' => $data->quantity[$key] * $data->price[$key] * 100,
                    'currency' => 'usd',
                    'customer' => $customer->id,
                    'payment_method' => $paymentMethod->id,
                    'capture_method' => 'manual',
                    'automatic_payment_methods' => [
                        'enabled' => true,
                        'allow_redirects' => 'never',
                    ],
                ]);
                
                

                

                $paymentIntent->confirm();
                $paymentIntent->capture();
    
                CustomerPayment::create([
                    'order_detail_id' => $order_datails_id,
                    'amount' => $data->quantity[$key] * $data->price[$key],
                    'payment_id' => $paymentIntent->id,
                    'payment_status' => "paid",
                    'source' => 'stripe',
                ]);
    
                Order::create([
                    'order_detail_id' => $order_datails_id,
                    'ad_id' => $ad_id,
                    'quantity' => $data->quantity[$key]
                ]);
            }
    
            // $this->OrderConfirmationMail($data);
            $this->sendOrderConfirmationMail($data);
            return redirect('/cart')->with(['success' => 'Your Order Has Been Placed successfully!', 'status' => 'success']);
    
        } catch (\Exception $e) {
            OrderDetail::where('id', $order_datails_id)->delete();
            return redirect('/cart')->with('error', $e->getMessage());
        }
    }
    

    // public function verify_cart(Request $request){
    //      $validated = $request->validate([
    //         'cart' => 'required|array',
    //         'cart.*.id' => 'required|integer', // Ensure each cart item has an ID
    //     ]);

    //     // Retrieve cart items sent from the frontend
    //     $cartItems = $validated['cart'];

    //     // Extract product IDs from the cart
    //     $cartProductIds = collect($cartItems)->pluck('id')->toArray();

    //     // Query the database to check if these products exist
    //     $existingProductIds = Ad::whereIn('id', $cartProductIds)->pluck('id')->toArray();

    //     // Filter out items that are not in the database
    //     $validCart = array_filter($cartItems, function ($item) use ($existingProductIds) {
    //         return in_array($item['id'], $existingProductIds);
    //     });

    //     return response()->json(['validCart' => $validCart]);
    // }

public function detailed_search(Request $request)
    {
        Log::info('Detailed search initiated', $request->all());
    
        // Initialize query for Ads
        $query = Ad::query();
    
        // Filter by title if provided
        if ($request->filled('title')) {
            $query->where('title', 'like', '%' . $request->title . '%');
            Log::info('Filtering by title', ['title' => $request->title]);
        }
    
        // Filter by type if provided
        if ($request->filled('type')) {
            $query->where('type', $request->type);
            Log::info('Filtering by type', ['type' => $request->type]);
        }
    
        // Filter by worn_days if provided
        if ($request->filled('worn_days')) {
            $query->where('worn_days', '<=', $request->worn_days);
            Log::info('Filtering by worn days', ['worn_days' => $request->worn_days]);
        }
    
        // Filter by price if provided
        if ($request->filled('price')) {
            $query->where('price', '<=', $request->price);
            Log::info('Filtering by price', ['price' => $request->price]);
        }
    
        // Filter by upload date if provided
        if ($request->filled('upload_date')) {
            $query->whereDate('created_at', $request->upload_date);
            Log::info('Filtering by upload date', ['upload_date' => $request->upload_date]);
        }
    
        // Filter by verified user profile
        $query->whereHas('user', function ($query) {
            $query->whereHas('profile', function ($query) {
                $query->where('status', 'verified');
            });
        });
    
        // Apply additional filters for status and ad status
        $query->where('status', 'Approved')
              ->where('ad_status', 'active')
              ->orderBy('id', 'desc');
    
        // Get the filtered ads
        $ads = $query->get();
        Log::info('Ads retrieved', ['count' => $ads->count()]);
    
        // Initialize query for media items
        $mediaItems = MediaFile::query();
    
        // Filter media items by title if provided
        if ($request->filled('title')) {
            $mediaItems->where('name', 'like', '%' . $request->title . '%');
            Log::info('Filtering media items by title', ['title' => $request->title]);
        }
    
        // Get the filtered media items
        $mediaItems = $mediaItems->get();
        Log::info('Media items retrieved', ['count' => $mediaItems->count()]);
    
        // Initialize query for categories
        $categories = Category::query();
    
        if ($request->filled('title')) {
            Log::info('Category search term', ['category' => $request->title]);  // Log title field as category search term
            $categories->where('category', 'like', '%' . $request->title . '%');
            Log::info('Filtering category items by category');
        } else {
            Log::info('No category filter applied');
        }
    
        // Get the filtered categories
        $categories = $categories->get();
        Log::info('category items retrieved', ['count' => $categories->count()]);
    
        // Return the results to the view
        return view('user.index', [
            'ads' => $ads, 
            'search' => 'search',
            'mediaItems' => $mediaItems,
            'categories' => $categories,
            'bannersliders' => [],
            'sectionrights' => []
        ]);
    }


    public function contact(Request $request){
        $request->validate([
            'name' => 'required|string',
            'email' => 'required|email',
            'subject' => 'required|string',
            'message' => 'required|string'
        ]);

        Contact::create([
            'name' => $request->name,
            'email' => $request->email,
            'subject' => $request->subject,
            'message' => $request->message
        ]); 

        return redirect()->back()->with(['success' => 'Your Message Has Been Sent successfully!']);
    }
    
}
