<?php

namespace App\Http\Controllers;

use App\Models\Ad;
use App\Models\CanRequest;
use App\Models\Category;
use App\Models\Contact;
use App\Models\User;
use App\Models\Can;
use App\Models\Profile;
use App\Models\MediaFile;
use App\Models\OrderDetail;
use App\Models\Payment;
use App\Models\ActivityLog;
use App\Models\Setting;
use App\Models\SellerActivityLog;
use App\Models\Report;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use App\Services\DynamicArchiveService;

class AdminController extends Controller
{

    protected $archiveService;

    public function __construct(DynamicArchiveService $archiveService)
    {
        $this->archiveService = $archiveService;
    }

    public function cleanup()
    {
        $this->archiveService->archiveOldRecords();
        return redirect()->back()->with('status', 'All old records archived successfully!');
    }

    public function showMediaLinks()
{
    $mediaLinks = MediaFile::with('seller')
    ->get(); // Get all links from the database
    return view('admin.mediafile', compact('mediaLinks'));
}

    public function showProfileSettings()
{
    $profileSettings = Profile::first(); // Assuming you have a single settings record for profile limits
    return view('admin.profilesettings', compact('profileSettings'));
}

public function updateProfileupdate(Request $request)
{
    $request->validate([
        'max_ads' => 'required|integer|min:1',
        'max_ads_premium' => 'required|integer|min:1',
        'max_media_per_ad' => 'required|integer|min:1',
        'max_media_per_ads_premium' => 'required|integer|min:1',
    ]);

    $settings = Profile::first(); // Update the settings
    $settings->max_ads = $request->max_ads;
    $settings->max_ads_premium = $request->max_ads_premium;
    $settings->max_media_per_ad = $request->max_media_per_ad;
    $settings->max_media_per_ads_premium = $request->max_media_per_ads_premium;
    $settings->save();

    return back()->with('success', 'Profile limits updated successfully.');
}

    public function adminlovecanwomen(Request $request)
{
    $unapprovedAds = Ad::where('is_lovecans', true)
                       ->where('admin_approved', false); // Unapproved Ads

    $approvedAds = Ad::where('is_lovecans', true)
                     ->where('admin_approved', true); // Approved Ads

    // Search functionality for unapproved ads
    if ($request->has('search') && !empty($request->search)) {
        $unapprovedAds->where('title', 'LIKE', '%' . $request->search . '%');
        $approvedAds->where('title', 'LIKE', '%' . $request->search . '%');
    }

    $unapprovedAds = $unapprovedAds->paginate(10);
    $approvedAds = $approvedAds->paginate(10);

    return view('admin.lovecanwomen', compact('unapprovedAds', 'approvedAds'));
}

    
    
    public function approveLovecansAd($id)
    {
        $ad = Ad::findOrFail($id);
        if ($ad->is_lovecans) {
            $ad->admin_approved = true;
            $ad->save();
            return redirect()->back()->with('success', 'Lovecans Ad Approved Successfully!');
        }
        return redirect()->back()->with('error', 'Invalid Ad!');
    }
    

    public function index(){
        $payments = Payment::all();
        return view('admin.dashboard',['payments'=>$payments]);
    }
    public function seller_profiles(Request $request)
{
    $query = User::query();

    // Filter: Only sellers with profiles
    $query->whereHas('profile', function ($q) {
        $q->whereIn('status', ['verified', 'pending', 'blocked']);
    });

    // Search by name
    if ($request->has('search') && $request->search != '') {
        $query->where('name', 'like', '%' . $request->search . '%');
    }

    // Filter: Role is seller
    $query->where('role', 'seller');

    $sellers = $query
        ->with('profile')
        ->select('users.*')
        ->paginate(50);

    // Return to view
    return view('admin.seller_profiles', ['sellers' => $sellers]);
}

public function selleractivitylog(Request $request)
{
    // Get the search term from the request
    $search = $request->input('search');

    // Query with conditional search if 'search' is provided
    $getaddress = SellerActivityLog::with('seller')
        ->when($search, function($query, $search) {
            return $query->whereHas('seller', function($query) use ($search) {
                $query->where('name', 'like', "%{$search}%");
            });
        })
        ->orderBy('activity_time', 'desc')
        ->paginate(5);

    // Pass the data to the view
    return view('admin.seller_activity_log', compact('getaddress'));
}


    public function verify_seller(){
        $verifications = DB::table('verifications')
                        ->join('users', 'users.id', '=', 'verifications.user_id')
                        ->join('profiles','users.id','profiles.user_id')
                        ->select('users.id as user_id', 'users.*', 'profiles.profile_picture', 'profiles.id as profile_id', 'verifications.file_path')
                        ->where('users.role', 'seller')
                        ->orderBy('user_id')
                        ->get()
                        ->groupBy('user_id'); // Group by user_id
        return view('admin.verify_seller', compact('verifications'));
    }
    
    public function pending_sellers(Request $request){
        
        $query = User::with(['profile' => function($query) {
            $query->where('status', 'pending'); 
        }]);
        if ($request->has('search') && $request->search != '') {
            $query->where('name', 'like', '%' . $request->search . '%');
        }
        $pending_sellers = $query
        ->has('profile')
        ->where('role', 'seller')
        ->select('users.*')
        ->paginate(50);
        return view('admin.pending_sellers',['data'=>$pending_sellers]);
    }
    public function locked_sellers(Request $request){
        $query = User::with(['profile' => function($query) {
            $query->where('status', 'blocked'); 
        }]);
        if ($request->has('search') && $request->search != '') {
            $query->where('name', 'like', '%' . $request->search . '%');
        }
        $locked_sellers = $query
        ->has('profile')
        ->where('role', 'seller')
        ->select('users.*')
        ->paginate(50);
        // return($locked_sellers );
        return view('admin.locked_sellers',['data'=>$locked_sellers]);
    }
    public function verified_sellers(Request $request){

        $query = User::with(['profile' => function($query) {
            $query->where('status', 'verified'); 
        }]);
        if ($request->has('search') && $request->search != '') {
            $query->where('name', 'like', '%' . $request->search . '%');
        }
        $verified_sellers = $query
        ->has('profile')
        ->where('role', 'seller')
        ->select('users.*')
        ->paginate(50);

        return view('admin.verified_sellers',['data'=>$verified_sellers]);
    }

    public function edit_seller($id){
        $data = DB::table('users')->join('profiles','users.id','profiles.user_id')
        ->select('users.*','profiles.id as profile_id','profiles.*')
        ->where('profiles.id',$id)
        ->get();
        return view('admin.edit_sellers',['data'=>$data]);
    }



    public function update_seller(Request $request){
        $request->validate([
            'name' => 'required',
            'email' => 'required|email',
            'age' => 'required|integer',
            'type'=> 'required|string',
            'figure' => 'required|string',
            'hair_color' => 'required|string',
            'eye_color' => 'required|string',
            'nickname' => 'required|string',
            'address' => 'required|string',
            'description' => 'required|string',
        ]);

        if($request->password){
            $request->validate([
                'password'=>'required|confirmed|min:8'
            ]);
            User::where('id',$request->user_id)->update(['password'=> Hash::make($request->password)]);
        }

        DB::table('users')
        ->where('id',$request->user_id)
        ->update([
            "name"=>$request->name,
            "email"=>$request->email
        ]);


        
        DB::table('profiles')
        ->where('id',$request->profile_id)
        ->update([
            'age' => $request->age,
            'type'=> $request->type,
            'figure' => $request->figure,
            'hair_color' => $request->hair_color,
            'eye_color' => $request->eye_color,
            'nickname' => $request->nickname,
            'address' => $request->address,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.seller_profiles');
    }

    public function cans(Request $request)
{
    $can = CanRequest::with('user')->whereHas('user', function($query) use ($request) {
        if ($request->has('search') && $request->search != '') {
            $query->where('name', 'like', '%' . $request->search . '%');
        }
    });
    
    $can = $can->orderBy('id', 'desc')->paginate(150);
    
    $totalCans = Can::sum('stock');

    return view('admin.cans', ['cans' => $can, 'totalCans' => $totalCans]);
}


    public function orders(){
        
        $orders = DB::table('ads')
        ->join('orders', 'orders.ad_id', '=', 'ads.id')
        ->join('order_details', 'orders.order_detail_id', '=', 'order_details.id')
        ->select('orders.*','orders.id as order_id','orders.status as order_status', 'ads.*', 'order_details.*')
        ->orderBy('orders.id', 'desc')
        ->paginate(150);
        return view('admin.orders',['orders'=>$orders]);
    }

    public function add_category(){
        return view('admin.add_category');
    }
    public function store_category(Request $request)
{
    // Validate the incoming request data
    $request->validate([
        'category' => 'required|string',
        'status' => 'required',
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // Validation for image
    ]);

    // Handle the image upload
    $imagePath = null;
    if ($request->hasFile('image')) {
        $imagePath = $request->file('image')->store('category_images', 'public'); // Save image to storage/app/public/category_images
    }

    // Create the category with the image path
    $result = Category::create([
        'category' => $request->category,
        'status' => $request->status,
        'image' => $imagePath, // Save the image path in the database
    ]);

    // Check if category creation was successful
    if (!$result) {
        return redirect()->route('admin.categories')->with('error', 'Couldn`t add category..! Please Try Again..!');
    }

    return redirect()->route('admin.categories')->with('success', 'Category added successfully');
}

    public function updateCans(Request $request)
{
    $validated = $request->validate([
        'quantity' => 'required|integer|min:1',
        'action' => 'required|in:add,remove',
    ]);

    $quantity = $validated['quantity'];
    $action = $validated['action'];

    $currentInventory = Can::sum('stock');

    if ($action == 'add') {
        $newInventory = $currentInventory + $quantity;
    } elseif ($action == 'remove' && $currentInventory >= $quantity) {
        $newInventory = $currentInventory - $quantity;
    } else {
        return back()->with('error', 'Cannot remove more cans than available.');
    }

    $can = Can::first();
    $can->update(['stock' => $newInventory]);

    return back()->with('success', 'Inventory updated successfully!');
}



    public function editcategory($id){
        $category = Category::find($id);
        return view('admin.edit-category', compact('category'));
    }

public function updatecategory(Request $request, $id)
    {
        // Find the category by its ID
        $category = Category::find($id);
    
        // Validate the request data
        $request->validate([
            'category' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // Optional image validation
        ]);
    
        // Check if there is a new image to upload
        if ($request->hasFile('image')) {
            // Delete the old image if it exists
            if ($category->image) {
                Storage::disk('public')->delete($category->image);
            }
    
            // Store the new image and get its path
            $imagePath = $request->file('image')->store('category_images', 'public');
        } else {
            // If no new image is uploaded, keep the old image path
            $imagePath = $category->image;
        }
    
        // Update the category
        $category->update([
            'category' => $request->category,
            'image' => $imagePath,
            'status' => $request->status,
        ]);
        return redirect()->route('admin.categories')->with('success', 'Category Updated successfully');
    }

    public function delete_category($id){
        $result = Category::where('id',$id)->delete();
        if(!$result){
            return redirect()->back()->with('error', 'Couldn`t delete category..! Please Try Again..!');
        }
        return redirect()->back()->with('success', 'Category deleted successfully');
    }

   public function reports(Request $request){
        
        $search = $request->search;
        $reports = Report::with('user', 'ad')
        ->when($search, function ($query) use ($search) {
            $query->where('report_reason', 'like', "%{$search}%")
            ->orWhere('report_details', 'like', "%{$search}%")
                ->orWhereHas('ad', function ($query) use ($search) {
                    $query->where('title', 'like', "%{$search}%");
                });
        })
        ->paginate(150);
        return view('admin.reports',compact('reports'));
    }

    public function get_customer_messages(Request $request){
        if ($request->has('search') && $request->search != '') {
            $messages = Contact::where('name', 'like', '%' . $request->search . '%')
            ->orderBy('id', 'desc')
            ->paginate(150);
        }else{
            $messages = Contact::orderBy('id', 'desc')
            ->paginate(150);
        }
        return view('admin.customer-messages',compact('messages'));
    }
    
    public function customerlist(Request $request)
    {
        $query = OrderDetail::query();
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
    
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%")
                  ->orWhere('phone_number', 'LIKE', "%{$search}%")
                  ->orWhere('address', 'LIKE', "%{$search}%")
                  ->orWhere('city', 'LIKE', "%{$search}%")
                  ->orWhere('postcode', 'LIKE', "%{$search}%");
            });
        }
        $customers = $query->orderBy('id', 'desc')->paginate(20);
        return view('admin.customerlist', compact('customers'));
    }

    public function customerlistupdate(Request $request, $id)
{
    $request->validate([
        'status' => 'required|in:unblock,blocked',  // 'add_notes' no longer needed here
        'notes' => 'nullable|string',  // Optional validation for notes
    ]);

    $customer = OrderDetail::find($id);

    if ($customer) {
        $customer->status = $request->input('status');

        // Check if notes are provided and save them
        if ($request->has('notes')) {
            $customer->notes = $request->input('notes');
        }

        $customer->save();

        return redirect()->back()->with('success', 'Customer status and notes updated successfully.');
    } else {
        return redirect()->back()->with('error', 'Customer not found.');
    }
}



    public function deletecustomerlist($id){
        $customers = OrderDetail::find($id);
        $customers->delete();
        return redirect()->back();
    }

    public function setting(){
        return view('admin.addsetting');
    }

    public function settings(Request $request)
    {
        $query = Setting::query();
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('promotion_fee', 'like', '%' . $search . '%')
                  ->orWhere('promotion_duration', 'like', '%' . $search . '%');
            });
        }
    
        $settings = $query->orderBy('created_at', 'desc')->paginate(10);
    
        return view('admin.settings', compact('settings'));
    }
    

    public function addSettings(Request $request)
    {

        $request->validate([
            'promotion_fee' => 'required',
            'promotion_duration' => 'required',
            'description' => 'nullable',
        ]);

            Setting::create([
                'promotion_fee' => $request->promotion_fee,
                'promotion_duration' => $request->promotion_duration,
                'description' => $request->description,
            ]);
    
        return redirect()->route('admin.settings')->with('success', 'Settings added successfully.');
    }

    public function editSettings($id){
        $settings = Setting::find($id);
        return view('admin.updatesettings', compact('settings'));
    }

    public function updateSettings(Request $request, $id){
        $settings = Setting::find($id);

        $request->validate([
            'promotion_fee' => 'required',
            'promotion_duration' => 'required',
        ]);

        $settings->update($request->all());

        return redirect()->route('admin.settings')->with('success', 'Settings updated successfully.');
    }

     public function deletesettings($id)
    {
        $setting = Setting::find($id);
        if (!$setting) {
            return back()->with('error', 'Setting not found.');
        }
        $setting->delete();
        return back()->with('success', 'Setting deleted successfully.');
    }

    public function showUserActivityLog(Request $request)
{
    $query = ActivityLog::with('user')->orderBy('created_at', 'desc');

    if ($request->has('search')) {
        $search = $request->search;
        $query->whereHas('user', function ($q) use ($search) {
            $q->where('name', 'LIKE', "%$search%");
        });
    }

    $activityLogs = $query->paginate(10);

    return view('admin.showUserActivityLog', compact('activityLogs'));
}


    public function logActivity($userId, $action, $details = null)
    {
        ActivityLog::create([
            'user_id' => $userId,
            'action' => $action,
            'details' => $details
        ]);
    }

    public function clearActivityLog($userId)
    {
    ActivityLog::where('user_id', $userId)->delete();

    $this->logActivity($userId, 'Deleted Log', 'Admin cleared activity log');

    return back()->with('success', 'Activity Log Cleared!');
    }

    public function gendermessages(Request $request)
    {
        $search = $request->input('search');
    
        $sellers = User::where('role', 'seller')
            ->whereHas('adminMessages')
            ->with(['profile', 'adminMessages'])
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%")
                      ->orWhereHas('profile', function ($q) use ($search) {
                          $q->where('gender', 'LIKE', "%{$search}%");
                      });
                });
            })
            ->paginate(10);
    
        return view('admin.gendermessges', compact('sellers'));
    }
    

    public function updateSellerGender(Request $request, $sellerId)
    {
        $request->validate([
            'gender' => 'required|in:male,female',
        ]);
    
        $seller = User::findOrFail($sellerId);
        $seller->update(['gender' => $request->gender]); // Direct update in users table
    
        return redirect()->back()->with('success', 'Seller gender updated successfully.');
    }


    public function get_all_notification()
{
    $users = User::where('status', 'pending')
    ->where('role', 'seller')
    ->latest()
    ->get();

    // Retrieve the latest products with 'Unapproved' status
    $products = Ad::where('status', 'Unapproved')
        ->latest()
        ->get();


    return response()->json([
        'users' => $users,
        'products' => $products,
    ]);
}

}
