<?php

namespace App\Http\Controllers;

use App\Models\Ad;
use App\Models\ActivityLog;
use App\Models\AdditionalRefinement;
use App\Models\Report;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;


class AdController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
{
    $currentDate = now();

    $ads = Ad::whereHas('sellerpayment', function($query) {
            $query->where('payment_status', 'Success');
        })
        ->whereHas('sellerpayment', function($query) {
            $query->where('duration', '>', 0);
        })
        ->get()
        ->filter(function($ad) use ($currentDate) {
            return $ad->sellerpayment && $ad->sellerpayment->created_at->addDays($ad->sellerpayment->duration)->isFuture();
        });

    return view("user.index", compact('ads'));
}

    

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view("seller.ad.create");
    }

    /**
     * Store a newly created resource in storage.
     */
     public function store(Request $request)
     {
         Log::info('Ad creation started.', ['user_id' => auth()->id()]);

         $user = auth()->user()->load('profile.membership');
         $membership = $user->profile->membership ?? (object) ['name' => 'Standard', 'type' => 'standard'];

         $today = now()->toDateString();
         $currentYear = now()->year;

         $premiumLimit = 8;
         $standardLimit = 1;

         $adsToday = $user->ads()->whereDate('created_at', $today)->count();
         $adsThisYear = $user->ads()->whereYear('created_at', $currentYear)->count();

         if ($membership->name === 'Premium' && $adsToday >= $premiumLimit) {
            return back()->with('error', 'As a Premium seller, you can only create 3 ad per day.');
         }

         if ($membership->name === 'Standard' && $adsThisYear >= $standardLimit) {
            return back()->with('error', 'As a Standard seller, you can only create 1 ad per day.');
         }

         $validated = $request->validate([
             'title' => 'required|string|max:255',
             'type' => 'required',
             'price' => 'required|integer',
             'worn_days' => 'required|integer|min:1|max:7',
             'description' => 'required|string|max:255',
             'image' => 'required|file|mimes:jpg,jpeg,png',
             'free_of_diseases' => 'required',
             'drugs_free' => 'required',
             'gallery' => 'array', 
             'gallery.*' => 'file|mimes:jpg,jpeg,png', 
         ]);
         $cleanDescription = $this->cleanText($request->input('description'));

         $adImage = time().'.'.$request->image->getClientOriginalExtension();
         $request->image->move(public_path('storage/ads'), $adImage);
         $validated['image'] = $adImage;

         // Gallery Images
         $imagePaths = [];
         if ($request->hasFile('gallery')) {
            foreach ($request->file('gallery') as $image) {
                $adImage = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension(); // Generate unique filenames
                $image->move(public_path('storage/ads'), $adImage);
                $imagePaths[] = $adImage;
            }
         }
         $validated['gallery'] = $imagePaths;

         $ad = new Ad($validated);
         $ad->user_id = auth()->id();

         if ($ad->save()) {
             $seller = auth()->user();
             $can = $seller->cans()->first();

             if ($can) {
                 $can->stock -= 1;
                 $can->save();
             }

             $ad_id = $ad->id;

             if($request->refinement_title) {
                foreach ($request->refinement_title as $key => $value) {
                    $additional = new AdditionalRefinement();
                    $additional->refinement_title = $value;
                    $additional->refinement_price = $request->refinement_price[$key] ?? 0;
                    $additional->ad_id = $ad_id;
                    $additional->save();
                }
            }
         }

         return redirect('/profile')->with('success', 'Ad created successfully!');
         
    }

    public function edit(String $id){
        $ad = Ad::findOrFail($id);
        return view('seller.ad.edit', compact('ad'));
    }

    public function update(Request $request, Ad $ad){
        Log::info('Ad update started.', ['ad_id' => $ad->id, 'user_id' => auth()->id()]);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required',
            'price' => 'required|integer',
            'worn_days' => 'required|integer|min:1|max:7',
            'description' => 'required|string|max:255',
            'image' => 'nullable|file|mimes:jpg,jpeg,png',
            'free_of_diseases' => 'required',
            'drugs_free' => 'required',
            'gallery' => 'array', 
            'gallery.*' => 'file|mimes:jpg,jpeg,png', 
        ]);

        $cleanDescription = $this->cleanText($request->input('description'));

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if (file_exists(public_path('storage/ads/' . $ad->image))) {
                unlink(public_path('storage/ads/' . $ad->image));
            }

            $adImage = time() . '.' . $request->image->getClientOriginalExtension();
            $request->image->move(public_path('storage/ads'), $adImage);
            $validated['image'] = $adImage;
        }

        $imagePaths = [];
        if ($request->hasFile('gallery')) {
            // Delete existing gallery images from storage
            if ($ad->gallery) {
                foreach ($ad->gallery as $oldImage) {
                    if (file_exists(public_path('storage/ads/' . $oldImage))) {
                        unlink(public_path('storage/ads/' . $oldImage));
                    }
                }
            }

            // Upload new gallery images
            foreach ($request->file('gallery') as $image) {
                $adImage = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('storage/ads'), $adImage);
                $imagePaths[] = $adImage;
            }
            $validated['gallery'] = $imagePaths;
        }


        $ad->fill($validated);

        if ($ad->save()) {
            $existingRefinements = $ad->additional_refinement->keyBy('refinement_title');
        
            if ($request->refinement_title) {
                foreach ($request->refinement_title as $key => $title) {
                    $price = $request->refinement_price[$key] ?? 0;
        
                    if (isset($existingRefinements[$title])) {
                        // Update existing refinement
                        $refinement = $existingRefinements[$title];
                        $refinement->refinement_price = $price;
                        $refinement->save();
                        unset($existingRefinements[$title]); // Remove from the list of refinements to delete
                    } else {
                        // Create new refinement
                        $additional = new AdditionalRefinement();
                        $additional->refinement_title = $title;
                        $additional->refinement_price = $price;
                        $additional->ad_id = $ad->id;
                        $additional->save();
                    }
                }
            }

            // Delete any remaining refinements that were not updated
            foreach ($existingRefinements as $refinement) {
                $refinement->delete();
            }
        }

        return redirect('/profile')->with('success', 'Ad updated successfully!');
    }
     
    private function cleanText($text)
    {
        $text = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $text);
        $text = preg_replace('/<iframe\b[^>]*>(.*?)<\/iframe>/is', '', $text);
        $text = preg_replace('/<object\b[^>]*>(.*?)<\/object>/is', '', $text);
        $text = preg_replace('/<embed\b[^>]*>(.*?)<\/embed>/is', '', $text);
        $text = preg_replace('/<applet\b[^>]*>(.*?)<\/applet>/is', '', $text);
        $text = preg_replace('/<meta\b[^>]*>/is', '', $text);

        $text = htmlspecialchars($text, ENT_QUOTES, 'UTF-8');

        $text = preg_replace('/on\w+="[^"]+"/i', '', $text);
        $text = preg_replace('/on\w+=\'[^\']+\'/i', '', $text);

        return trim($text);
    }
     

function logActivity($userId, $action, $details = null)
    {
        ActivityLog::create([
            'user_id' => $userId,
            'action' => $action,
            'details' => $details
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Ad $ad)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, $id)
{
    $ad = Ad::find($id);

    if (!$ad) {
        return response()->json([
            'message' => 'Ad not found.',
            'redirect_url' => route('ad.return-back-failed')
        ], 404);
    }

    $request->validate([
        'password' => 'required'
    ]);

    if (!Hash::check($request->password, auth()->user()->password)) {
        return response()->json([
            'message' => 'Invalid password.',
            'redirect_url' => route('ad.return-back-failed')
        ], 403);
    }

    $ad->delete();

    return response()->json([
        'message' => 'Ad deleted successfully.',
        'redirect_url' => route('ad.return-back')
    ]);
}


    public function return_back(){
        return redirect()->route('seller.dashboard')->with('success', 'Ad delete successfully!');

    }
    public function return_back_failed(){
        return redirect()->route('seller.dashboard')->with('error', 'Invalid password!');

    }
    


    public function report(Request $request, $product_id){
       
        $validated = $request->validate([
            'report_id'=> 'required|string',
            'report_details'=> 'required',
        ]);

        $result = Report::create([
            'ad_id' => $product_id,
            'report_id' => $validated['report_id'],
            'report_details' => $request->report_details
        ]);
    
        if($result){
            return redirect()->back()->with('success', 'Report sent successfully');
        }
        return redirect()->back()->with('error', 'Failed to send report');
    }

    public function ad_status_update(Request $request, $ad_id){
        $result = Ad::where('id', $ad_id)->update([
            'ad_status' => $request->ad_status
        ]);

        if($result){
            return redirect()->back()->with('success', 'Ad status updated successfully');
        }
        return redirect()->back()->with('error', 'Failed to update ad status');
    }

    public function togglePause($adId)
{
    $ad = Ad::findOrFail($adId);

    // Toggle the 'paused' status
    $ad->paused = !$ad->paused;
    $ad->save();

    return redirect()->back()->with('status', 'Ad status updated successfully!');
}

public function promote($id){
    $ad = Ad::findOrFail($id); // Fetching one ad

    return view('seller.promote', compact('ad')); // Passing a single ad
}
    
}
